<xsl:stylesheet 
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform" 
  xmlns:svg="http://www.w3.org/2000/svg"
  xmlns:doc="http://nwalsh.com/xsl/documentation/1.0"
  xmlns:msxsl="urn:schemas-microsoft-com:xslt"
  exclude-result-prefixes="doc"
  extension-element-prefixes="msxsl"
  version="1.0"
>

<!-- insertsvgfont.xsl 

     XSL stylesheet to introduce the character glyphs referenced by
     the SVG text elements in the input file into at the beginning
     between <defs>...</defs> elements.

     Principle

     First a linear pass is made over the data to locate the "g"
     elements having a style attribute. From that the font name and
     the character size are determined. Then, from the enclosed text
     element we get the codes of the characters referenced. All this
     information is written into a "char" element, as follows:

     <char fontname="fn" filename"fn10.svg" code="code"/> 

     During this first pass over all lines in the file we store the
     character "char" elements in a variable "chars-rtf". 
     This variable must then transformed into a node-set again, from
     which we then construct a new variable, for which, with the kelp of
     the "chars" key we sort the elements by font and character name.
     We also only take the first occurrence of a given pair 
     ("charssorted-rtf" and "charssorted" variables). 

     These variables are next used to extract the necessary glyph
     images from the svg font files in the svg subdirectory .

     All this happens in the "getchars" named template.

     Finally, we copy the rest of the input file (everything is
     encapsulated in "g" elements to the output.

     First version mod.  8 February 2002 (MG)

  -->

<xsl:output method='xml' indent='yes' encoding='iso-8859-1'/>

<xsl:strip-space elements="*"/>

<!-- key for sorting and uniquing in charssorted-rtf variable -->
<xsl:key name="chars" match="char"
         use="concat(@fontname, '+', @code)" />

<!-- key to loop over font names -->
<xsl:key name="fn" match="char" use="@fontname"/>

<!--############################################################################# -->
<!-- Write identification and get started                                         -->
<!--############################################################################# -->

<xsl:variable name="STYLEVERSION" select="1.1"/>
<xsl:variable name="LASTMOD">
  <xsl:text>11 Feb. 2002 (MG)</xsl:text>
</xsl:variable>
<xsl:variable name="xsl-vendor" select="system-property('xsl:vendor')"/>

<xsl:template match="/">
<xsl:message>################################################################################	</xsl:message>
<xsl:message> XSLT stylesheet to enter SVG fonts into TeX-generated SVG file		</xsl:message>
<xsl:message> 								</xsl:message>
<xsl:message> VERSION  : <xsl:value-of select="$STYLEVERSION"/>		</xsl:message>
<xsl:message> LAST MOD : <xsl:value-of select="$LASTMOD"/>		</xsl:message>
<xsl:message>  AUTHOR  : Michel Goossens (michel.goossens@cern.ch)	</xsl:message>
<xsl:message>   USING  : <xsl:value-of select="$xsl-vendor"/>		 </xsl:message>
<xsl:message>################################################################################	</xsl:message>
     <xsl:apply-templates select="svg"/>
</xsl:template>

<!--############################################################################# -->
<!-- Main template (for root element svg)                                         -->
<!--       Writes svg and title.                                                  -->
<!--       Calls getchars                                                         -->
<!--       Copies everything that follows in input stream                         -->
<!--############################################################################# -->

<xsl:template match="svg">
 <xsl:element name="svg">
  <xsl:attribute name="width"><xsl:value-of select="@width"/></xsl:attribute>
  <xsl:attribute name="height"><xsl:value-of select="@height"/></xsl:attribute>
  <xsl:if test="@viewBox">
    <xsl:attribute name="viewBox"><xsl:value-of select="@viewBox"/></xsl:attribute>
  </xsl:if>
  <xsl:text>&#xA;</xsl:text>
  <xsl:element name="title"><xsl:value-of select="//title"/></xsl:element>
  <xsl:text>&#xA;</xsl:text>
  <xsl:call-template name="getchars-msxsl"/>
  <xsl:apply-templates select="g"/>
 </xsl:element>
</xsl:template>

<!--############################################################################# -->
<!-- Templates for copy through of input stream                                   -->
<!--       Ignore title (already copied)                                          -->
<!--       Copy elements and attributes of all <g> containers.                    -->
<!--############################################################################# -->

<xsl:template match="title"/>

<xsl:template match="g">
<xsl:copy-of select=".|@*"/>
</xsl:template>

<!--############################################################################# -->
<!-- Getchars: Main worker to extract the font and char names from the input      -->
<!--           and get the relevant glyph info into the output file               -->
<!--       1) Gather information into variable $chars-rtf                         -->
<!--          It performs this task by extracting the information from the        -->
<!--          style attribute of g elements and from the content of the enclosed  -->
<!--          text elements get the glyph codes                                   -->
<!--       2) Transform the RTF into a nodeset (variable $chars)                  -->
<!--       3) sort the font/glyph pairs, and using a combination of key           -->
<!--          and getid eliminate duplicates. The result goes in $charssorted-rtf -->
<!--       4) Transform the RTF into a nodeset (variable $charssorted)            -->
<!--       5) Open a <defs> element                                               -->
<!--            For each fontname: open a font and fontface element               -->
<!--              For each referenced glyph in that font                          -->
<!--                Copy the glyph from the relevant font file to the output      -->
<!--            Close fontface and font elements                                  -->
<!--          Close  <defs> element                                               -->
<!--############################################################################# -->

<xsl:template name="getchars-msxsl">

<!-- First pass over the data: store chars in chars-rtf -->
  <xsl:variable name="chars-rtf">
   <xsl:element name="fontlist">
    <xsl:apply-templates select="g[@style]" mode="getchars"/>
   </xsl:element>
  </xsl:variable>

  <!-- Turn result tree fragment into a node set -->
  <xsl:variable name="chars"
                select="msxsl:node-set($chars-rtf)//char" />
<!-- End of first pass over the data. The variable $chars now contains
     the nodeset with all char elements (unsorted)
  -->

<!-- Sort the characters, eliminate duplicates, and store RTF
     in variable charssorted-rtf -->
  <xsl:variable name="charssorted-rtf">
   <xsl:element name="fontlist">
  <xsl:apply-templates  mode="getchars"
    select="$chars[generate-id() =
                   generate-id(key('chars',
                                   concat(@fontname, '+', @code))[1])]">
    <xsl:sort select="@fontname" />
    <xsl:sort select="@code" />
    </xsl:apply-templates>
   </xsl:element>
  </xsl:variable>

  <!-- Turn result tree fragment into a node set -->
  <xsl:variable name="charssorted"
                select="msxsl:node-set($charssorted-rtf)/fontlist/char" />
  
<!-- End of sorting and unification process, the variable charssorted
     contains a sorted list containing a font/char pair for each
     distinct character in the input file 
  -->

<!-- We are now going to loop over these pairs, and for each font we create 
     a font element with the name of the font (generating font and
     font-face elements). Then we start a second loop over all the
     characters in that font, copying for each desired code the glyph
     present in the file in the svg directory with the name specified. 
  -->

<xsl:variable name="fontlist">
  <xsl:value-of select="document('svg/fontlist.xml')/fontlist"/> 
</xsl:variable>

  <xsl:element name="defs">
  <!-- Outer loop (over fonts) -->
    <xsl:for-each select="$charssorted[generate-id() =
                        generate-id(key('fn',@fontname)[1])]">
  <!-- check whether that font is present in the directory -->
    <xsl:if test="contains($fontlist, concat(@fontname,'/'))">
     <xsl:element name="font">
      <xsl:attribute name="id">
        <xsl:value-of select="@fontname"/>
      </xsl:attribute>
      <xsl:element name="font-face">
        <xsl:attribute name="font-family">
          <xsl:value-of select="@fontname"/>
        </xsl:attribute>
      </xsl:element>
    <!-- Inner loop (over characters in given font) -->
      <xsl:for-each select="key('fn',@fontname)">
        <xsl:variable name="code" select="@code"/>
        <xsl:variable name="filename">
          <xsl:text>svg/</xsl:text>
          <xsl:value-of select="@filename"/>
        </xsl:variable>
      <!-- copy character glyph from external file to present file -->
        <xsl:copy-of
           select="document($filename)/font/glyph[@unicode=$code]"/> 
      </xsl:for-each>
     </xsl:element>
    </xsl:if> <!-- if font not present, do nothing -->
    </xsl:for-each>
  </xsl:element>
</xsl:template>

<!--############################################################################# -->
<!-- Helper template for getchars named template.                                 -->
<!--        It corresponds to stage 1) in the description of getchars:            -->
<!--       1a) extract into variable $font the font name encoded into the         -->
<!--           style attribute of g elements that contain text elements.          -->
<!--       1b) extract into variable $text the character data enclosed in the     -->
<!--           <text>...</text> element                                           -->
<!--       1c) Call the "printchar" named template                                -->
<!--           This will generate one <char> element for each character inside    -->
<!--           the text element, i.e., if there are n characters, then n <char>   -->
<!--           are generated, and they contain each the font name, a filename     -->
<!--           and the code of the character                                      -->
<!--############################################################################# -->

<xsl:template match="g[@style]" mode="getchars">
  <xsl:if test="contains(@style,'font-family')">
  <xsl:variable name="font"
  select="substring-before(substring-after(@style,'font-family:'),';')"/>
  <xsl:variable name="text" select="./text"/>
  <xsl:call-template name="printchar">
    <xsl:with-param name="chars" select="$text"/>
    <xsl:with-param name="font" select="$font"/>
  </xsl:call-template>
  </xsl:if>
</xsl:template>

<!--############################################################################# -->
<!-- Helper named template "printchar"                                            -->
<!--        It corresponds to stage 1) in the description of getchars:            -->
<!--        There are two parameters:                                             -->
<!--          chars: one or more character codes (from inside a <text> element)   -->
<!--          font : the name of the font from where these character codes have   -->
<!--                 to take they pgyph representations                           -->
<!--                                                                              -->
<!--          printchar is a recursive template                                   -->
<!--                                                                              -->
<!--          the first character of the $chars parameter is copied to the local  -->
<!--          variable $char                                                      -->
<!--                                                                              -->
<!--          For each character code a <char> empty element is generated         -->
<!--          It has the following three attributes:                              -->
<!--           fontname: name of the font (uppercased and with digits eliminated) -->
<!--           filename: name of the file where font glyph data resides           -->
<!--                     (lowercased, digits eliminated, "10.svg" appended)       -->
<!--           code    : code of the character (as found in the <text> element)   -->
<!--                                                                              -->
<!--          If the length of the $chars parameters is > 1, there are still      -->
<!--          characters left to treat, and we call "printchar" another time      -->
<!--          with the $chars parameter set to the input $chars minus the first   -->
<!--          character that we already handled.                                  -->
<!--                                                                              -->
<!--          If the length of the $chars parameters is = 1 we finish             -->
<!--                                                                              -->
<!--############################################################################# -->

<xsl:template name="printchar">
  <xsl:param name="chars"/>
  <xsl:param name="font"/>
  <xsl:variable name="char" select="substring($chars,1,1)"/>
  <xsl:text>&#xA;</xsl:text>
  <xsl:element name="char">
    <xsl:attribute name="fontname">
     <xsl:value-of select="translate($font,
                                     'abcdefghijklmnopqrstuvwxyz0123456789',
                                     'ABCDEFGHIJKLMNOPQRSTUVWXYZ')"/>
    </xsl:attribute>
    <xsl:attribute name="filename">
     <xsl:value-of select="translate($font,
                                     'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789',
                                     'abcdefghijklmnopqrstuvwxyz')"/>
     <xsl:text>10.svg</xsl:text>
    </xsl:attribute>
    <xsl:attribute name="code">
     <xsl:value-of select="$char"/>
    </xsl:attribute>
   </xsl:element>
  <xsl:variable name="count" select="string-length($chars)"/>
  <!-- if more than one character in text element, we shift the
       contents in the char variable by one and call the present
       template recursively, until we have done the last character
       (lenght in $chars is 1) 
    -->
  <xsl:if test="$count &gt; 1">
    <xsl:call-template name="printchar">
      <xsl:with-param name="chars" select="substring($chars,2)"/>
      <xsl:with-param name="font" select="$font"/>
    </xsl:call-template>
  </xsl:if>
</xsl:template>


<!--############################################################################# -->
<!-- Helper template for getchars named template.                                 -->
<!--        It corresponds to stage 3) in the description of getchars:            -->
<!--        It copies through all the char elements and their attributes          -->
<!--        that pass the test in the foreach loop                                -->
<!--############################################################################# -->

<xsl:template match="char" mode="getchars">
     <xsl:copy-of select=".|@*"/>
</xsl:template>

</xsl:stylesheet>
